const { MongoClient, ServerApiVersion } = require("mongodb");
const { PingAPI } = require('@influxdata/influxdb-client-apis');
const { InfluxDB } = require('@influxdata/influxdb-client')
const pewModule = require('./pew_module');
const path = require('path');
const pew = new pewModule.pewModule();
const tls = require("tls");
const fs = require("fs");
const moduleName = "no-sqlClient Connection test";
//Catch exceptions
process.on('uncaughtException', function (err) {
    pew.sysLogMessage(moduleName, "Uncaught exception: " + err.stack);
})

var nosqlConfig = pew.getConfFileSync(pew.Constants.requests.READ_NOSQL);
if (nosqlConfig.err) {
    return console.log("Read configuration error");
}
var secureContext;

if (nosqlConfig.data.ca !== "" && nosqlConfig.data.client !== "" && nosqlConfig.data.key !== "") {
    secureContext = tls.createSecureContext({
        ca: fs.readFileSync(`${__dirname}/config/nosql/${nosqlConfig.data.ca}`),
        cert: fs.readFileSync(`${__dirname}/config/nosql/${nosqlConfig.data.client}`),
        key: fs.readFileSync(`${__dirname}/config/nosql/${nosqlConfig.data.key}`),
    });
}

const dbTypes = {
    mongodb: 0,
    influx: 1
}

pew.getKeyContent().then(key => {
    pew.Constants.pwkey = key;
    try {
        switch (nosqlConfig.data.db_type) {
            case dbTypes.mongodb:
                let uri = nosqlConfig.data.server;
                if (uri.search("<username>") !== -1) {
                    uri = uri.replace("<username>", nosqlConfig.data.user);
                }
                if (uri.search("<password>") !== -1) {
                    uri = uri.replace("<password>", pew.decryptPW(nosqlConfig.data.pass));
                }
                const dbInstance = nosqlConfig.data.instance_name;
                let client;
                //Check if authentication via Cert
                if (uri.search("MONGODB-X509") !== -1) {
                    const credentials = path.join(__dirname, pew.Constants.configSubfolders.nosql, nosqlConfig.data.client);
                    client = new MongoClient(uri, {
                        tlsCertificateKeyFile: credentials,
                        serverApi: ServerApiVersion.v1
                    });
                }
                else {
                    client = new MongoClient(uri, {
                        serverApi: {
                            version: ServerApiVersion.v1,
                            strict: true,
                            deprecationErrors: true,
                        },
                        tls: nosqlConfig.data.use_encryption === 1,
                        tlsAllowInvalidHostnames: nosqlConfig.data.ca_verification === 0,
                        secureContext
                    });
                }                
                runTestMongo(dbInstance, client).catch(err => console.log(err));
                break;

            case dbTypes.influx:
                let url = nosqlConfig.data.server;
                let token = pew.decryptPW(nosqlConfig.data.pass);

                const clientinflux = new InfluxDB({
                    url, token, transportOptions: {
                        rejectUnauthorized: nosqlConfig.data.ca_verification === 0 ? false : true
                    }
                })
                const pingAPI = new PingAPI(clientinflux);
                pingAPI.getPing().then(() => {
                    console.log(pew.Constants.API_TEST_SUCCESS);
                }).catch(err => console.log(err.message));
                break;
        }
    }
    catch (ex) {
        console.log(ex.message);
    }
})

async function runTestMongo(instance, client) {
    try {
        // Connect the client to the server (optional starting in v4.7)
        await client.connect();
        // Send a ping to confirm a successful connection
        await client.db(instance).command({ ping: 1 });
        console.log(pew.Constants.API_TEST_SUCCESS);
    } finally {
        // Ensures that the client will close when you finish/error
        await client.close();
    }
}
